#!/bin/bash
echo "starting script..."

set -euo pipefail

# Setup kubeconfig file
export KUBECONFIG=~/kubeconfig.yaml
echo "$VAR_kubeconfig" >> ~/kubeconfig.yaml
chmod 600 "$KUBECONFIG"


# Check if a Helm release exists in an optional namespace
helm_release_exists() {
    local release=$1
    local namespace=$2
    helm status "$release" -n "$namespace" >/dev/null 2>&1
}


if [ "${ACTION}" == "destroy" ]; then

    echo "Running destroy commands..."

        # Uninstall Helm chart
        helm uninstall my-httpbin -n ${NAMESPACE}

        # Delete the namespace entirely
        kubectl delete namespace ${NAMESPACE} &

        echo "Waiting for namespace ${NAMESPACE} to be deleted..."

        # Loop until namespace no longer exists
        while kubectl get namespace "${NAMESPACE}" > /dev/null 2>&1; do
                echo "Namespace ${NAMESPACE} still exists... waiting 5 seconds"
                sleep 5
        done

	echo "Namespace '$NAMESPACE' has been deleted."


elif [ "${ACTION}" == "deploy" ]; then

    echo "Starting deployment..."
	
	if helm_release_exists "my-httpbin" ${NAMESPACE}; then
		echo "[OK] Helm Chart already installed."
	else
		echo "[INFO] Installing Helm Chart..."
		
		# -----------------------------
        # Create Namespace
        # -----------------------------

        echo "Checking namespace..."

		if kubectl get namespace "${NAMESPACE}" >/dev/null 2>&1; then
			echo "[OK] Namespace '${NAMESPACE}' already exists. Skipping."
		else
			echo "[INFO] Namespace '${NAMESPACE}' not found. Creating..."
			kubectl create namespace "${NAMESPACE}"
		fi

		# -----------------------------
        # Update Helm Values
        # -----------------------------
		echo "Updating Helm Values..."
		envsubst < /helm/values.yaml > /tmp/values.yaml
		cat /tmp/values.yaml
		
		# -----------------------------
        # Install Helm Chart
        # -----------------------------
		echo "Installing Helm Chart..."
		helm install my-httpbin /helm/httpbin \
          --namespace "$NAMESPACE" \
          --wait \
          --timeout 5m \
          -f /tmp/values.yaml
	fi
		  
		  
	# -----------------------------
	# Get Replica Count
	# -----------------------------
	echo "Getting replica count from Kubernetes..."

	ACTUAL_REPLICA_COUNT=$(kubectl get deployment my-httpbin -n "$NAMESPACE" -o jsonpath='{.spec.replicas}')
	echo "Replica Count: $ACTUAL_REPLICA_COUNT"
	
	# -----------------------------
	# Output Values
	# -----------------------------

	echo ""
	echo "============================"
	echo "Helm Chart Deployed!"
	echo "============================"
	echo "Replica Count:"
	echo "$ACTUAL_REPLICA_COUNT"
	echo ""

	# Output file name
	output_file="output.json"

    # Create JSON content with the concatenated value
json_content=$(cat <<EOF
{
  "Replica_Count": "$ACTUAL_REPLICA_COUNT"
}
EOF
)


	# Write the JSON content to the file
	echo "$json_content" > "$output_file"

	# Confirm creation
	echo "JSON file '$output_file' created with content:"
	cat "$output_file"

	AUTH_TOKEN=$DRIVER_UPLOAD_TOKEN
	API_ENDPOINT=$DRIVER_UPLOAD_URL

	if [[ "${DRIVER_SKIP_TLS_VERIFY:-false}" == "true" ]]; then
			echo "insecure"
			curl  \
			 -X POST \
			 -k -F "content=@output.json" -H "X-Engine-Helper-Token: $AUTH_TOKEN" --max-time 60 \
			   --retry 3 --retry-all-errors $API_ENDPOINT     
	else
			curl  \
			 -X POST \
			 -F "content=@output.json" -H "X-Engine-Helper-Token: $AUTH_TOKEN" --max-time 60 \
			  --retry 3 --retry-all-errors  $API_ENDPOINT     
	fi
else
    echo "Invalid action: $action"
    echo "Usage: $0 [deploy|destroy]"
    exit 1
fi